---
name: code-refactoring
description: Provides targeted refactoring recommendations, performance and safety impact analysis, and produces concrete code patches and test suggestions for Java and Kotlin projects (enterprise and Android). Triggered by requests to simplify class/ module design, improve thread-safety, optimize slow loops, reduce complexity, or produce patchable diffs and test cases.
allowed-tools: []
---

Skill purpose

This Skill helps engineers refactor Java and Kotlin code to improve performance, safety (thread-safety, null-safety), and maintainability. It analyzes provided code snippets or repositories, recommends concrete refactorings with an explanation of performance and risk trade-offs, generates patch-style code changes (unified diff or file replacement), and proposes unit/integration tests to validate the change.

Step-by-step instructions Claude must follow

1. Input intake
   - Ask for the minimal necessary context if not provided: language (Java or Kotlin), project type (enterprise backend, Android), build system (Maven/Gradle), target files or repository link, and specific concerns (e.g., slow loop, concurrency bug, large class).
   - Request tests or reproduce steps if a runtime issue is reported.

2. Static analysis & quick triage
   - Read the supplied code or repository files. If a full repo is provided, identify entry points and modules relevant to the reported issue.
   - Run a mental checklist: algorithmic complexity hotspots (nested loops, O(n^2) patterns), blocking I/O on main threads (Android), shared mutable state without synchronization, improper use of volatile/synchronized, incorrect use of mutable static fields, nullable handling, and large single-responsibility violations.

3. Propose prioritized refactor options
   - Produce 2–4 alternative refactor approaches ordered by risk and expected benefit: quick low-risk changes (local variable immutability, final fields, simple caching), moderate changes (replace synchronized blocks with concurrent collections or locks), and high-impact changes (algorithm replacement, API redesign, offloading to worker threads or coroutines).
   - For each option provide: short description, estimated performance impact (qualitative and approximate numeric estimate when possible, e.g., reduce O(n^2) to O(n log n) or expected 5–10x speed-up for N>1000), safety/bug-risk changes, required code areas, and an estimated effort level.

4. Generate concrete code patches
   - For the selected option, produce a unified diff or file-level replacement snippets for Java/Kotlin that can be applied with git apply. Include only necessary context lines and a clear header path for each changed file.
   - Ensure code follows typical Java/Kotlin style (null-safety idioms, final/val usage, proper exception handling), and include brief inline comments when non-obvious.

5. Produce tests and validation steps
   - Provide unit tests (JUnit 5 or Android JUnit/Instrumented tests / KotlinTest as appropriate) that specifically validate correctness and performance where feasible (benchmarks using JMH or simple microbench tests with warmup and iteration counts). Include input ranges and expected outputs.
   - Provide integration test suggestions and manual repro steps if the issue is runtime/environment-specific.

6. Risk assessment and rollback guidance
   - Describe potential regressions and unsafe assumptions introduced by the refactor (threading, ordering, memory) and list smoke tests to detect regressions.
   - Provide a minimal rollback patch or guidance to revert changes if problems appear.

7. Deliver final summary
   - Provide a concise summary of the chosen change, expected benefits, commands to apply patches (git apply path/to/patch), test commands (mvn test, gradle test, ./gradlew connectedAndroidTest), and next recommended steps.

Usage examples

Example 1 — Optimize slow loop
- Input: A Java method with nested loops iterating over two lists causing O(n*m) behavior with N up to tens of thousands.
- Output: Identify algorithmic improvement (use hashmap index or sort-merge join), give estimated complexity improvement (from O(n*m) to O(n+m) or O(n log n)), provide unified diff that replaces nested loop with HashMap-driven lookup, and include JUnit tests covering edge cases and performance microbenchmark instructions.

Example 2 — Improve thread-safety in Android
- Input: Kotlin Android class using lazy initialization without synchronization accessed from multiple threads.
- Output: Explain race condition risk, propose usage of lazy(mode = LazyThreadSafetyMode.SYNCHRONIZED) or volatile double-checked locking, provide Kotlin patch, and include instrumentation test or multi-threaded unit test to reproduce fix.

Example 3 — Simplify large service class
- Input: Enterprise Java service with many responsibilities and complex stateful logic.
- Output: Suggest extract-class/extract-method refactors, present staged diffs to move responsibilities into smaller classes, propose unit tests per new component, and outline migration and rollback plan.

Best practices

- Always request the smallest reproducible example if a full repo is not available.
- Prefer smallest safe change that yields meaningful benefit. When in doubt, produce a low-risk patch and an optional higher-impact patch separately.
- Provide estimates for benefits but mark them as approximations; recommend benchmarking in the target environment.
- Include tests that assert correctness first, then performance expectations second.
- When concurrency is involved, prefer proven concurrency utilities (java.util.concurrent, Kotlin coroutines) over ad-hoc synchronization.

Links to related files/scripts

- If repository-level changes are requested and sample scripts or benchmarks are generated, reference them using relative paths in the patches (e.g., ./benchmarks/BenchmarkHello.java or ./tests/refactor/RefactorTest.kt). Create those files only when requested.

Examples of commands to run after applying patches

- Apply patch: git apply path/to/patch.diff
- Run unit tests (Maven): mvn -DskipITs=false test
- Run Gradle tests: ./gradlew test
- Run Android instrumented tests: ./gradlew connectedAndroidTest

Notes and placeholders

- If the user does not specify language or project type, assume Java (enterprise) as default and ask follow-up clarifying questions before producing diffs.

