---
name: code-flow-web-doc-generator
description: Generates a web-document (HTML) that explains program behavior with visual diagrams (call flow, data flow) and concise node-level descriptions; triggered by requests to analyze codebases, files, or functions for quick structural understanding.
---

# Skill: Code Flow Web Document Generator

Purpose

This Skill analyzes source code (one or more files, modules, or a provided repository snippet) and produces a single self-contained web document (HTML) that helps reviewers and maintainers quickly understand the program's overall structure and runtime flow. The document emphasizes diagrammatic overviews (call sequences and data flow) plus concise explanations for each node (function/module): role, inputs, outputs, side effects, and important implementation notes.

Step-by-step instructions (must-follow)

1. Input intake
   - Accept either: a) a single file, b) multiple files, c) a repository path/zip, or d) pasted code snippets and an entry point (main function or primary endpoint).
   - If an entry point is not specified, attempt to infer it (e.g., main() in many languages, exported handler in server code); ask the user to confirm if ambiguous.

2. Static analysis and extraction
   - Parse the code to extract functions, classes, imports/exports, and module boundaries. Use language-appropriate parsing heuristics when exact parsing is not available.
   - Identify call relationships (who calls whom) by scanning invocations, method references, and exported usage sites. Mark direct calls, callbacks, and event-driven links separately.
   - Identify data flow: trace major data objects/structures passed between functions or modules (e.g., request -> handler -> service -> DAO). Summarize primary data shapes where feasible.
   - Extract short summaries for each function/module: one-line role, inputs (params/types if available), outputs/return values, side effects (I/O, DB, network), complexity or notable patterns.

3. Diagram generation
   - Produce at least two diagrams: a) High-level module/component diagram showing modules and their interactions, b) Call sequence / flow diagram showing primary call paths from the entry point(s) to key sinks (DB, network, file I/O).
   - Use a textual diagram format compatible with common renderers (Mermaid) so the output HTML can render diagrams client-side. Provide fallbacks to SVG/PNG if rendering is unavailable.
   - Group nodes logically (by module, package, or file). Annotate arrows with brief labels (e.g., "auth token", "userId", "callback").

4. Web document construction
   - Create a single HTML file with the following sections: Title & metadata, Table of Contents, High-level summary, Diagrams (interactive if Mermaid used), Node details (one block per function/module), Important call traces (numbered brief flows), Potential hotspots/maintenance notes, and How to run/inspect locally (commands or suggestions).
   - Include code excerpts (3–12 lines) near each node description where helpful. Do not include entire files unless requested.
   - Ensure the HTML is self-contained (inline CSS and scripts) so it can be opened locally.

5. Quality checks and user follow-up
   - Validate diagram references match node names and ensure no orphan nodes without descriptions. If ambiguity remains (dynamic calls, reflection), add an "Unresolved/Assumptions" section describing limits.
   - If the codebase is large, create a configurable depth/limit (e.g., top N functions, or depth D from entry point). If limits are applied, document them at the top.
   - Provide a short CLI snippet or command to regenerate the document with different options.

Usage examples

- Example 1: Single-file script
  - Input: a Python script with main() and helper functions.
  - Output: HTML showing main -> parser -> process -> save call sequence, Mermaid sequence diagram, node blocks for main, parse_args, process_data, save_results.

- Example 2: Multi-module web service
  - Input: repository snippets: api/handlers.py, services/users.py, db/dao.py, with entry point api/handlers.login_handler.
  - Output: Module diagram (api -> services -> db), flow from HTTP request -> handler -> service -> DAO, node details (inputs: HTTP request, outputs: HTTP response), side effects (sets session cookie).

- Example 3: Event-driven code
  - Input: event consumer and handler modules.
  - Output: Diagram labeling event source -> consumer -> handler -> external API, with arrows labeled by event type.

Best practices

- Provide an entry point when possible; it reduces ambiguity and improves diagram accuracy.
- For large codebases, request a focused slice (specific modules or feature) and set a depth limit to keep diagrams readable.
- Prefer static languages or well-structured dynamic code for more precise analysis; where dynamic features are used, list assumptions in the document.
- Use Mermaid diagrams for interactive diagrams in the HTML; include an option to export diagrams as PNG for presentations.

Common triggers (when to use this Skill)

- "Create a web document that explains this code and shows call/data flow"
- "Generate a diagram + explanations for these files and the main entry point"
- "I need a one-page overview for code review and onboarding"

Outputs

- A single self-contained HTML file named code-flow-documentation.html that includes diagrams (Mermaid), node descriptions, code excerpts, and a short how-to-run or regenerate section.

Limitations and assumptions

- Dynamic dispatch, runtime code generation, and heavy reflection may produce incomplete call graphs; report such limitations explicitly in the generated HTML.
- When precise type information is not available, use inferred types or describe shapes in plain language.

Related links and tools (optional)

- Use Mermaid (https://mermaid-js.github.io) for diagrams embedded in HTML.
- Suggest using language parsers (tree-sitter, AST modules) when available for higher accuracy.

Examples of commands to regenerate

- Regenerate with depth limit 2 and entry point `app.main`:
  - generate-doc --input ./src --entry app.main --depth 2 --output code-flow-documentation.html


