---
name: async-await-keyword-fixer
description: Identifies and fixes missing async/await keywords in JavaScript/TypeScript. Use when code uses await, Promise-returning functions, or async control flow incorrectly due to missing async or await, causing syntax errors or unintended behavior. Provides safe, minimal, keyword-only edits and clear explanations without introducing new logic or dependencies.
---
# Async/Await Keyword Fixer (JS/TS)

## Purpose

Help correct JavaScript/TypeScript code that uses `async`/`await` but is missing required keywords (e.g., `await` inside a non-`async` function, missing `await` on a promise, missing `async` on callbacks). The goal is **minimal, safe edits** that add or adjust **keywords only**, improving correctness without changing intent.

## Safety and Security Constraints

* Do **not** execute code, run commands, access the network, or interact with files.
* Do **not** add new logic, libraries, dependencies, or side effects.
* Do **not** refactor structure beyond what is needed to place `async`/`await` correctly.
* Do **not** handle secrets or request sensitive information.
* Keep changes strictly to **adding/removing/moving** the keywords `async` and/or `await`, and only when necessary to restore correctness.

## When to Activate

Use this Skill when the user:

* Reports errors like **“await is only valid in async functions”**, **SyntaxError** around `await`, or unexpected promise behavior.
* Shows code where:

  * `await` appears inside a function lacking `async`.
  * A promise-returning call is used as if it were a resolved value (likely missing `await`).
  * An async callback is required (e.g., in `map`, `forEach`, event handlers), but `async` is missing.
  * Control flow expects sequential behavior but omits `await`.

## What to Produce

Provide:

1. A concise diagnosis of what keyword is missing and why.
2. A minimal set of edits that fix it (keyword-only changes).
3. A brief note on any remaining semantic pitfalls **without** proposing refactors.

## Process

### 1) Scan for Common Keyword Omissions

Look for these patterns:

* **`await` used in a non-`async` function**

  * Any function body containing `await` must have `async` on the nearest enclosing function (unless it’s at top-level in an environment that supports top-level await).

* **Missing `await` on a promise-returning expression**

  * Calls like `fetch()`, `response.json()`, database/network calls, or any function the user indicates returns a `Promise`.
  * Assignments or returns used as if resolved:

    * `const x = someAsyncCall()` then `x.value` suggests missing `await`.

* **Missing `async` on callbacks that contain `await`**

  * Arrow functions, inline callbacks, and handler functions:

    * `arr.map((x) => { await ... })` → callback must be `async`.
    * `element.addEventListener('click', () => { await ... })` → handler must be `async`.

* **Loops intending sequential awaits**

  * `for` / `for...of` with promise calls inside may be missing `await`.
  * Do not refactor; only add `await` if the code is clearly treating results as resolved.

### 2) Determine the Minimal Correct Placement

Rules:

* If `await` exists inside a function, add `async` to that function declaration/expression.
* If a promise-returning call is used where a resolved value is required, add `await` at the call site.
* Prefer **the smallest scope**: mark only the function that directly contains `await` as `async` (not ancestors) unless the structure requires propagating `async` due to return type expectations.

### 3) Respect TypeScript Implications (Without Refactoring)

* Adding `async` changes the return type to `Promise<T>`.
* If the surrounding code expects `T`, it may require adding `await` at the call site of this function.
* Apply the smallest keyword-only change to align the expectation:

  * Either make the callee `async` (if it uses `await`), or add `await` where it’s called (if the caller expects a value).

### 4) Output Format Guidance

* Do not include large rewrites.
* Provide a short “Edit list” using natural language (e.g., “Add `async` to the function `foo`” / “Add `await` before `bar()`”).
* If the user supplied code, point to exact locations by function name and statement description.

## Common Fix Patterns

* `await` inside function → **add `async` to that function**
* promise used as value → **add `await` at usage site**
* callback containing `await` → **add `async` to callback**
* function became `async` and callers break → **add `await` at call sites that need resolved values**

## Pitfalls to Warn About (Briefly)

* `Array.prototype.forEach` does not await async callbacks; this may cause unexpected ordering.

  * Do not refactor; only mention as a caution if relevant.
* `map(async ...)` returns an array of promises; consumers may need to `await` them (e.g., via `Promise.all`).

  * Do not introduce `Promise.all`; only note the implication.

## Examples of User Requests This Skill Should Handle

* “Why do I get ‘await is only valid in async functions’?”
* “My function uses await but I forgot something—can you fix it?”
* “This code returns a promise when I expected a value.”
* “My event handler uses await and crashes.”

## Quality Bar

* Changes must be minimal, keyword-only, and clearly justified.
* Never add new APIs, control flow constructs, or restructure code.
* Keep the explanation practical and focused on the missing `async`/`await` keywords.
