---
name: code-commentary-applier
description: Inserts beginner-friendly explanatory comments into existing source code files by editing the files directly while preserving behavior. Use when the user explicitly asks to apply comments into files, annotate files in-place, or save commented versions back to the repository.
allowed-tools: Read, Write, Grep, Glob
---
# Code Commentary Applier

## Purpose

Apply explanatory comments **directly into source code files** (in-place edits) to improve readability for humans—especially junior developers—without changing program behavior.

This Skill is for **file modification**, not just generating commented text.

## Hard Safety Rules

1. **Never change behavior**

   * Do not refactor, rename identifiers, reorder logic, alter control flow, or change data structures.
   * Do not add, remove, or modify executable statements.
   * Do not add or remove imports, dependencies, build configs, or tooling.
   * Only insert comments and (when idiomatic and already used in the file) doc comments/docstrings.

2. **Least-change editing**

   * Make the smallest possible edit set: comments only, minimal whitespace impact.
   * Preserve existing formatting and line endings as much as possible.
   * Do not reformat entire files.

3. **No sensitive information in comments**

   * If the code contains secrets (tokens, passwords, private keys, internal hosts, personal data), do not repeat them.
   * Use placeholders like `<REDACTED>` / `<SECRET>` in any explanatory comment.

4. **No risky guidance**

   * Do not add comments that suggest bypassing security, disabling checks, weakening auth, or ignoring verification/errors.

5. **No external links**

   * Do not include URLs in inserted comments.

## When This Skill May Edit Files

Only proceed with writing changes when **both** are true:

* The user explicitly requested file modification (e.g., “apply comments to the file”, “edit files in-place”, “save changes”).
* The target files are clearly identified (paths or discoverable by unambiguous patterns).

If either condition is not met, do not write—limit to analysis and request the needed file targets.

## Tooling Policy

* Use **Glob** to locate candidate files when the user provides a directory or pattern.
* Use **Grep** to find key symbols and best insertion points.
* Use **Read** to inspect full context before editing.
* Use **Write** only after preparing a safe, comment-only change plan.

## Workflow

1. **Identify targets**

   * Confirm the exact file paths (or resolve with Glob).
   * Avoid editing generated files, vendored dependencies, minified bundles, lockfiles, or build artifacts unless the user explicitly requests it.

2. **Read and map structure**

   * Locate entry points, exported/public APIs, complex logic, error paths, and invariants.
   * Determine the file’s existing comment conventions and follow them.

3. **Plan comment insertions**

   * Prioritize:

     * Module/file purpose (brief header if missing and clearly inferable)
     * Public functions/classes (parameters/returns/error behavior)
     * Non-obvious logic (regexes, bitwise ops, concurrency, caching)
     * Data shape and invariants
     * Error handling and retries
   * Avoid:

     * Trivial narration (“increment i”)
     * Restating the code verbatim
     * Speculative intent (“this was added because…”)

4. **Apply edits safely**

   * Insert comments near the relevant code blocks.
   * Keep comments short, plain, and accurate.
   * Maintain indentation and alignment.

5. **Validate non-interference**

   * Re-check that no executable code changed.
   * Ensure comments do not break syntax (e.g., inside strings, between tokens, or within sensitive constructs).

## Comment Style Guide

* Match the file’s language and comment syntax.
* Use neutral phrasing: “Does X” / “Handles Y”.
* If uncertain, prefer: “This appears to…” and keep it minimal.
* Document gotchas as “Watch out:” and focus on common pitfalls.

## Quality Bar

Edits should make the code easier to understand while remaining:

* Correct
* Minimal
* Secure
* Consistent with existing project style

## Allowed Outputs

When this Skill is active, the final response should:

* Briefly list which files were edited and what types of comments were added (high level).
* Avoid pasting large portions of source code unless the user asks.
* Never include secrets or sensitive values.

## Refusal Conditions

Refuse (or limit to read-only guidance) if the request requires any of the following:

* Changing logic, behavior, performance characteristics, or architecture
* Adding features, tests, or new files
* Inserting comments that reveal or recreate sensitive data
* Producing or embedding program code beyond comment insertions in existing files
